###
### backmon.lib.environment
###

import sys
import os.path
import glob
import time

import backup_monitoring.builtins

from backup_monitoring.parsing.exceptions import ParseError
from backup_monitoring.parsing.routines import strip_domain

from backup_monitoring.parsing.parsers import nbemmcmd
from backup_monitoring.parsing.parsers import bpstulist
from backup_monitoring.parsing.parsers import df
from backup_monitoring.parsing.parsers import dsu_ls_l
from backup_monitoring.parsing.parsers import nbstlutil
from backup_monitoring.parsing.parsers import bpdbjobs
from backup_monitoring.parsing.parsers import nbdevquery
from backup_monitoring.parsing.parsers import vmpool
from backup_monitoring.parsing.parsers import vmquery
from backup_monitoring.parsing.parsers import bpmedialist
from backup_monitoring.parsing.parsers import bpplclients
from backup_monitoring.parsing.parsers import bpretlevel
from backup_monitoring.parsing.parsers import bppllist

from .exceptions import BackmonError
from .updates import *

class Environment(object):

    def __init__(self, name, logger, updates, master, media, tz=None):

        self.name               = name
        self.logger             = logger
        self.tz                 = tz
        self.alias              = name.lower()
        self.master             = master
        self.media_servers      = media
        self.updates            = ExtendedDict()
        self.buffers            = ExtendedDict()
        self.aliases            = ExtendedDict()
        self.stunits            = ExtendedDict()
        self.df_data            = ExtendedDict()
        self.dsu_contents       = ExtendedDict()
        self.lifecycle_images   = ExtendedDict()
        self.jobs               = ExtendedDict()
        self.disk_pools         = ExtendedDict()
        self.pools              = ExtendedDict()
        self.volumes            = ExtendedDict()
        self.media              = ExtendedDict()
        self.scratch_pool       = ''
        self.clients            = ExtendedDict()
        self.retlevels          = ExtendedDict()
        self.policies           = ExtendedDict()

        self.logger.debug('name=%s, alias=%s, master=%s' % (self.name, self.alias, self.master))

        for server in self.media_servers:
            self.logger.debug('media=%s' % (server))
        
        if not os.path.isdir(updates):
            raise BackmonError('Updates directory %s does not exist!' % (updates))
        else:
            for path in glob.glob(os.path.join(updates, '*')):
                server = os.path.basename(path)
                self.logger.debug('+ %s -> %s' % (server, path))
                self.updates[server] = Updates(logger, path)

    #
    # feed() - wrapper interface to server updates objects feed methods
    #
    def feed(self, server, feed):

        try:

            return self.updates[server].feed(feed)

        except Exception, e:

            raise BackmonError, e

    #
    # buffer() - buffer feed in memory
    #
    def buffer(self, server, feed):

        try:

            buffer = '%s.%s' % (server, feed)
            self.logger.debug('buffering %s...' % (buffer))
            self.buffers[buffer] = self.feed(server, feed)

        except Exception, e:

            raise BackmonError, e
        
    #
    # refresh() - refresh buffers
    #
    def refresh(self, objects=[]):

        try:

            master = self.master

            if 'aliases' in objects:

                self.buffer(master, 'nbemmcmd_machinealias_getaliases')

            if 'jobs' in objects:

                self.buffer(master, 'bpdbjobs_most_columns')

        except Exception, e:

            raise BackmonError, e

    #
    # load_feeds() - load a set of input feeds in preparation for parsing
    #
    def load_feeds(self, master=[], media=[], client=[]):

        self.logger.debug('%s - loading feeds..' % (self.name))

        try:

            server = self.master

            for feed in master:
                self.buffer(server, feed)

            for feed in media:
                for server in self.media_servers:
                    self.buffer(server, feed)

        except Exception, e:

            raise BackmonError, e

    #
    # parse_aliases() - parse aliases
    #
    def parse_aliases(self):

        self.logger.debug('%s - parsing aliases..' % (self.name))

        master = self.master

        try:

            buffer = '%s.nbemmcmd_machinealias_getaliases' % (master)

            if buffer in self.buffers:

                self.aliases = ExtendedDict()

                stream = nbemmcmd.stream(self.buffers[buffer], format='nbemmcmd -machinealias -getaliases')

                for record in stream:

                    alias, aliases = nbemmcmd.parse(record, format='nbemmcmd -machinealias -getaliases', tz=self.tz)

                    self.aliases[alias] = aliases

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # resolve_alias() - resolve alias to server receiving updates
    #
    def resolve_alias(self, alias):

        try:

            if alias not in self.aliases:
                return strip_domain(alias)
            else:
                for server in self.aliases[alias]:
                    if server in self.updates:
                        return server

            return strip_domain(alias)

        except Exception, e:
            raise BackmonError, e

    #
    # parse_stunits() - parse storage units
    #
    def parse_stunits(self):

        self.logger.debug('%s - parsing stunits..' % (self.name))

        master = self.master

        try:

            buffer = '%s.bpstulist' % (master)

            if buffer in self.buffers:

                self.stunits = ExtendedDict()

                stream = bpstulist.stream(self.buffers[buffer])

                for record in stream:

                    stunit = bpstulist.parse(record, tz=self.tz)

                    #self.logger.debug(stunit.label)

                    self.stunits[stunit.label] = stunit

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_df_data() - parse df data
    #
    def parse_df_data(self):

        self.logger.debug('%s - parsing df_data..' % (self.name))

        servers = [self.master]
        servers.extend(self.media_servers)

        try:

            self.df_data = ExtendedDict()

            for server in servers:

                if server not in self.df_data:

                    self.df_data[server] = ExtendedDict()

                    buffer = '%s.df' % (server)

                    if buffer in self.buffers:

                        stream = df.stream(self.buffers[buffer])

                        for record in stream:

                            fs = df.parse(record, tz=self.tz)

                            self.logger.debug('%s:%s' % (server, fs.mountpoint))

                            self.df_data[server][fs.mountpoint] = fs

                        del self.buffers[buffer]

                    else:

                        raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_dsu_contents() - parse disk storage unit contents (from ls -l)
    #
    def parse_dsu_contents(self):

        self.logger.debug('%s - parsing dsu_contents..' % (self.name))

        servers = self.media_servers

        try:

            self.dsu_contents = ExtendedDict()

            for server in servers:

                if server not in self.dsu_contents:
                    self.dsu_contents[server] = ExtendedDict()

                    buffer = '%s.dsu_ls_l' % (server)

                    if buffer in self.buffers:

                        stream = dsu_ls_l.stream(self.buffers[buffer])

                        for record in stream:

                            dir = dsu_ls_l.parse(record, tz=self.tz)

                            self.logger.debug('%s:%s' % (server, dir.path))

                            self.dsu_contents[server][dir.path] = dir

                        del self.buffers[buffer]

                    else:

                        raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_lifecycle_images() - parse images processed by storage lifecycles
    #
    def parse_lifecycle_images(self):

        self.logger.debug('%s - parsing lifecycle_images..' % (self.name))

        master = self.master

        try:

            buffer = '%s.nbstlutil_list' % (master)

            if buffer in self.buffers:

                self.lifecycle_images = ExtendedDict()

                stream = nbstlutil.stream(self.buffers[buffer], format='nbstlutil list -U')

                for record in stream:

                    image = nbstlutil.parse(record, format='nbstlutil list -U', tz=self.tz)
                    image_id = image.backup_id

                    #self.logger.debug(image_id)

                    self.lifecycle_images[image_id] = image

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_jobs() - parse jobs
    #
    def parse_jobs(self):

        self.logger.debug('%s - parsing jobs..' % (self.name))

        master = self.master

        try:

            buffer = '%s.bpdbjobs_most_columns' % (master)

            if buffer in self.buffers:

                self.jobs = ExtendedDict()

                stream = bpdbjobs.stream(self.buffers[buffer])

                for record in stream:

                    job = bpdbjobs.parse(record, tz=self.tz)

                    #self.logger.debug(job.jobid)

                    self.jobs[job.jobid] = job

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_disk_pools() - parse disk_pools
    #
    def parse_disk_pools(self):

        self.logger.debug('%s - parsing disk pools..' % (self.name))

        master = self.master

        self.disk_pools = ExtendedDict()

        for feed in ['nbdevquery_listdv_stype_basicdisk', 'nbdevquery_listdv_stype_advanceddisk', 'nbdevquery_listdv_stype_puredisk']:

            try:

                buffer = '%s.%s' % (master, feed)

                if buffer in self.buffers:

                    stream = nbdevquery.stream(self.buffers[buffer])

                    for record in stream:

                        if len(record) == 1 and record[0] == 'no disk volumes found!':
                            break

                        pool = nbdevquery.parse(record, tz=self.tz)

                        #self.logger.debug(pool.disk_pool_name)

                        self.disk_pools[pool.disk_pool_name] = pool

                    del self.buffers[buffer]

                else:

                    raise BackmonError, 'buffer %s is empty!' % (buffer)

            except ParseError, e:
                raise BackmonError, e

    #
    # parse_pools() - parse volume manager pools
    #
    def parse_pools(self):

        self.logger.debug('%s - parsing pools..' % (self.name))

        master = self.master

        try:

            buffer = '%s.vmpool' % (master)

            if buffer in self.buffers:

                self.pools = ExtendedDict()

                stream = vmpool.stream(self.buffers[buffer])

                for record in stream:

                    pool = vmpool.parse(record, tz=self.tz)

                    self.logger.debug(pool.pool_name)

                    self.pools[pool.pool_name] = pool

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_scratch() - parse volume manager scratch pool
    #
    def parse_scratch(self):

        self.logger.debug('%s - parsing scratch pool..' % (self.name))

        master = self.master

        try:

            buffer = '%s.vmpool_list_scratch' % (master)

            if buffer in self.buffers:

                self.pools = ExtendedDict()

                stream = vmpool.stream(self.buffers[buffer], format='vmpool -list_scratch')

                for record in stream:

                    pool = vmpool.parse(record, format='vmpool -list_scratch', tz=self.tz)

                    self.logger.debug(pool)

                    self.scratch_pool = pool

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e


    #
    # parse_volumes() - parse volume manager volumes
    #
    def parse_volumes(self):

        self.logger.debug('%s - parsing volumes..' % (self.name))

        master = self.master

        try:

            buffer = '%s.vmquery' % (master)

            if buffer in self.buffers:

                self.volumes = ExtendedDict()

                stream = vmquery.stream(self.buffers[buffer])

                i = 0

                for record in stream:

                    i += 1

                    volume = vmquery.parse(record, tz=self.tz)

                    if i % 100 == 0:
                        self.logger.debug(i)

                    self.volumes[volume.media_id] = volume

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_media() - parse media list
    #
    def parse_media(self):

        self.logger.debug('%s - parsing media..' % (self.name))

        master = self.master

        try:

            buffer = '%s.bpmedialist' % (master)

            if buffer in self.buffers:

                self.media = ExtendedDict()

                stream = bpmedialist.stream(self.buffers[buffer])

                i = 0

                for record in stream:

                    i += 1

                    media = bpmedialist.parse(record, tz=self.tz)

                    if i % 100 == 0:
                        self.logger.debug(i)

                    self.media[media.media_id] = media

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_clients() - parse clients
    #
    def parse_clients(self):

        self.logger.debug('%s - parsing clients..' % (self.name))

        master = self.master

        try:

            buffer = '%s.bpplclients' % (master)

            if buffer in self.buffers:

                self.clients = ExtendedDict()

                stream = bpplclients.stream(self.buffers[buffer])

                for record in stream:

                    client = bpplclients.parse(record, tz=self.tz)

                    #self.logger.debug(client.name)

                    self.clients[client.name] = client

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_retlevels() - parse retention levels
    #
    def parse_retlevels(self):

        self.logger.debug('%s - parsing retlevels..' % (self.name))

        master = self.master

        try:

            buffer = '%s.bpretlevel' % (master)

            if buffer in self.buffers:

                self.retlevels = ExtendedDict()

                stream = bpretlevel.stream(self.buffers[buffer])

                for record in stream:

                    retlevel = bpretlevel.parse(record, tz=self.tz)

                    #self.logger.debug(retlevel.retention_level)

                    self.retlevels[retlevel.retention_level] = retlevel

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e

    #
    # parse_policies() - parse policies
    #
    def parse_policies(self):

        self.logger.debug('%s - parsing policies..' % (self.name))

        master = self.master

        try:

            buffer = '%s.bppllist' % (master)

            if buffer in self.buffers:

                self.policies = ExtendedDict()

                stream = bppllist.stream(self.buffers[buffer])

                for record in stream:

                    policy = bppllist.parse(record, tz=self.tz)

                    #self.logger.debug(policy.policy_name)

                    self.policies[policy.policy_name] = policy

                del self.buffers[buffer]

            else:

                raise BackmonError, 'buffer %s is empty!' % (buffer)

        except ParseError, e:
            raise BackmonError, e
