###
### Temporal Functions
###

import time
import datetime
import calendar
import pytz

from ..math import micropad

__all__ = [
    'timedelta_to_seconds',
    'mindatetime',
    'maxdatetime',
    'hh_mm_ss',
    'dd_hh_mm',
    'dd_hh_mm_ss',
    'timeslot',
    'backup_day',
    'iso',
]

##
## TimeError
##
class TimeError(StandardError):
    pass


##
## Return the duration of a timedelta object in seconds as a float.
##
def timedelta_to_seconds(timedelta):
    return float('%s.%s' % ((timedelta.days * 86400 ) + (timedelta.seconds),  micropad(timedelta.microseconds)))


##
## Minimum date + 1 to allow for timezones
##
def mindatetime(tz=pytz.timezone('UTC')):
    return datetime.datetime(datetime.MINYEAR, 1, 2, tzinfo=tz)


##
## Maximum date + 1 to allow for timezones
##
def maxdatetime(tz=pytz.timezone('UTC')):
    return datetime.datetime(datetime.MAXYEAR, 12, 30, 23, 59, 59, 999999, tzinfo=tz)


##
## Convert seconds to HH:MM:SS
##
def hh_mm_ss(seconds):
    minutes, seconds = divmod(seconds, 60)
    hours, minutes = divmod(minutes, 60)
    return '%02d:%02d:%02d' % (hours, minutes, seconds)


##
## Convert seconds to DD:HH:MM
##
def dd_hh_mm(seconds):
    minutes, seconds = divmod(seconds, 60)
    hours, minutes = divmod(minutes, 60)
    days, hours = divmod(hours, 24)
    return '%02d:%02d:%02d' % (days, hours, minutes)


##
## Convert seconds to DD HH:MM:SS
##
def dd_hh_mm_ss(seconds):
    minutes, seconds = divmod(seconds, 60)
    hours, minutes = divmod(minutes, 60)
    days, hours = divmod(hours, 24)
    return '%02d %02d:%02d:%02d' % (days, hours, minutes, seconds)


##
## Round timestamp to timeslot according to step value
##
def timeslot(t, step=60):

    if 86400 % step != 0:
        raise TimeError, 'step %d is not evenly divisible into 86400 seconds (1 day)!' % (step)
    elif type(t) is datetime.datetime:
        seconds = (t.minute * 60) + t.second 
        seconds = seconds - seconds % step
        minute, second = divmod(seconds, 60)
        return t.replace(minute=minute, second=second, microsecond=0)
    elif type(t) is int or type(t) is long:
        return t - t % step
    elif type(t) is float:
        t = long(t)
        return t - t % step
    else:
        return None
        

##
## Return backup day from datetime
##
def backup_day(t, offset=8):

    if t.hour < offset:
        t = t - datetime.timedelta(days=1)

    return t.replace(hour=0, minute=0, second=0, microsecond=0)


##
## Return timestamp as ISO formatted string
##
def iso(t,nosep=False):

    if nosep is True:
        return t.strftime('%Y%m%d%H%M%S')
    else:
        return t.strftime('%Y-%m-%d %H:%M:%S')


