import re
import pytz
import dateutil
import dateutil.parser

from ....builtins import *

from ...exceptions import *
from ...expressions import *
from ...autotype import *
from ...streams import *

from .. import bpplclients

re_cannot_connect_on_socket = re.compile('^cannot connect on socket.*$')

##
## bppllist has records separated by blank lines
##
def stream(stream, format='bppllist -L'):

    if format in ['bppllist -L']:
        return BlankLineStream(stream, header=1)
    else:
        raise ParseError, 'Unknown format %s' % (format)


##
## Parse a bppllist record
##
##     bppllist -L
##
def parse(record, format='bppllist -L', version=None, tz=None):

    re_pair = re.compile('^\s*([^:]+):\s*(.*)$')
    re_type_id = re.compile('^(.+)\s+\((\d+)\)$')
    re_nbu_datetime = re.compile('^.*\d\d:\d\d:\d\d\s+\((\d+)\)$')

    policy = ExtendedDict()

    clients = ExtendedDict()
    includes = []
    excludes = []

    policy['clients'] = clients
    policy['includes'] = includes
    policy['excludes'] = excludes

    if re_cannot_connect_on_socket.match(record[0]):
        return None

    if format == 'bppllist -L':

        header = True

        try:

            i = 0

            #
            # process policy header
            #
            while header and i < len(record):

                line = record[i]

                match = re_pair.match(line)

                key = match.group(1)
                value = match.group(2)
                key = key.lower()
                key = key.replace(' ', '_')
                key = key.replace('.', '')
                key = key.replace('/', '_')

                if key == 'schedule':
                    header = False
                else:

                    #
                    # match timestamp with Unix time in parentheses
                    #
                    match = re_nbu_datetime.match(line)
                    if match:
                        value = datetime.datetime.fromtimestamp(float(match.group(1)), tz)

                    #
                    # clients (entire line can be parsed with bpplclients parser)
                    #
                    elif key == 'client_hw_os_pri':
                        client = bpplclients.parse(line)
                        clients[client.client] = client
                        key = None

                    #
                    # no clients defined
                    #
                    elif key == 'clients' and value == '(none defined)':
                        value = []

                    #
                    # options
                    #
                    elif key == 'options':
                        value = int(value, 16)  # convert to integer from hexadecimal string

                    #
                    # includes
                    #
                    elif key == 'include':
                        includes.append(value)
                        key = None

                    #
                    # excludes
                    #
                    elif key == 'exclude':
                        if value != '(none defined)':
                            excludes.append(value)
                        key = None

                    #
                    # effective date does not include a Unix timestamp
                    #
                    elif key == 'effective_date':
                        value = dateutil.parser.parse(value).replace(tzinfo=tz)

                    #
                    # separate type and id
                    #
                    elif re_type_id.match(value):
                        match = re_type_id.match(value)
                        idkey = '%s_id' % (key)
                        value = match.group(1)
                        policy[idkey] = int(match.group(2))

                    #
                    # values to replace with Python None datatype
                    #
                    elif value in ['(none)', '(none specified)', '(none defined)']:
                        value = None

                    #
                    # ..otherwise autodetect
                    #
                    else:
                        value = autotype(value)

                    #
                    #
                    #
                    if key is not None:
                        policy[key] = value

                    i += 1

            #
            # parse schedules
            #
            schedules = ExtendedDict()

            while i < len(record):

                line = record[i]

                match = re_pair.match(line)

                #
                # this is typical key:value metadata
                #
                if match:

                    key = match.group(1)
                    value = match.group(2)
                    key = key.lower()
                    key = key.replace(' ', '_')
                    key = key.replace('-', '_')
                    key = key.replace('/', '_')

                    #
                    # identify current schedule
                    #
                    if key == 'schedule':

                        schedule = ExtendedDict()

                        schedule['schedule'] = value

                        schedules[value] = schedule

                    #
                    # type and type_id
                    #
                    elif key == 'type':
                        re_type_id.match(value)
                        match = re_type_id.match(value)
                        idkey = '%s_id' % (key)
                        value = match.group(1)
                        schedule[idkey] = int(match.group(2))

                    #
                    # daily windows
                    #
                    elif key == 'daily_windows':

                        daily_windows = ExtendedDict()

                        re_daily_window_labels = re.compile('\s+')
                        re_daily_window = re.compile('\s+')

                        i += 1
                        line = record[i]
                        labels = []
                        for label in re_daily_window_labels.split(line):
                            if label != '':
                                label = label.lower()
                                label = label.replace('-','_')
                                labels.append(label)
                        label_count = len(labels)

                        i += 1
                        for line in record[i:i+7]:

                            daily_window = ExtendedDict()

                            items = []
                            for item in re_daily_window.split(line):
                                if item != '':
                                    item = item.lower()
                                    items.append(item)

                            item_count = len(items)
                            for index in range(label_count):
                                label = labels[index]
                                if index < item_count:
                                    item = items[index]
                                else:
                                    item = None
                                daily_window[label] = item

                            daily_windows[daily_window.day] = daily_window

                        i += 7

                        value = daily_windows

                    #
                    # values to replace with Python None datatype
                    #
                    elif value in ['(none)', '(none specified)', '(none defined)']:
                        value = None

                    #
                    # ..otherwise autodetect
                    #
                    else:
                        value = autotype(value)

                    #
                    #
                    #
                    if key is not None:
                        schedule[key] = value

                #
                # this is not typical key:value metadata
                #
                else:
                    pass

                i += 1

            policy['schedules'] = schedules

            return policy

        except Exception, e:

            for line in record:
                print line

            raise ParseError, e

    else:

        raise ParseError, 'Unknown format %s' % (format)

