#!/bin/bash 
# build script
#
# $Id: build.sh 735 2009-01-26 15:45:04Z joergs $
#
# builds RPM in an clean chroot build environment.
# uses the Suse build script.
#
# without parameter, it checks the directory for the files
# SOURCES or SPECSOURCES. 
# The files definied in there are copied to the 
# build system.
# 
# when used with parameter, 
# this parameter is expected to be a source rpm
#
# config file:
# /usr/local/etc/build
# lists the sources for the different distributions
#
# the rest is controlled by environment variables
#
# build.sh uses sudo. /etc/sudoers needs a line like:
# %users  ALL=(ALL)       NOPASSWD: /usr/local/bin/buildscript
# for automatic building (create a symlink or copy the file,
# see also the Variable BUILDSCRIPT and content of this script)
#
# return codes:
# 0: success
# 1: skipped (current RPM already in dest dir)
# >1: error
#
# RPM releases are set 
# to the svn version number of the project directory.
# Modifiers are:
#   m: release + local modifications
#   p: release + submited modifications. svn update project directory is needed 


# set LANG to default (english)
# required for SVN
export LC_ALL=C

CONFIG_FILE=/usr/local/etc/build
BUILDSCRIPT=/usr/local/bin/buildscript
BUILD_CMD=/usr/bin/build
BUILDUSER=$USER

[ -z "$BUILD_MODE" ] && BUILD_MODE="--clean"
[ -z "$BUILD_DIST" ] && BUILD_DIST="sles9-i386"
[ -z "$BUILD_ROOT" ] && BUILD_ROOT="/var/tmp/buildsystem.$USER.$BUILD_DIST"
[ -z "$BUILD_PREPARE_SPEC" ] && BUILD_PREPARE_SPEC="no"

[ -z $DEST_DIR ] && DEST_DIR="/tmp/build.$BUILDUSER"
BUILDTEMP=$DEST_DIR/buildsrc/
BUILDVARS=$BUILDTEMP/build.vars

[ -z "$BUILD_USES_SVN" ] && BUILD_USES_SVN="yes"



if [ -r $CONFIG_FILE ]; then
	source $CONFIG_FILE
    MOD_BUILD_DIST=`sed "s/[-+.]/_/g" <<< $BUILD_DIST`
    echo MOD_BUILD_DIST: $MOD_BUILD_DIST
    eval BUILD_RPMS_TEST=\$BUILD_RPMS_$MOD_BUILD_DIST

	if [ "$BUILD_RPMS_TEST" ]; then
		if [ "$BUILD_RPMS" ]; then
			BUILD_RPMS=$BUILD_RPMS:$BUILD_RPMS_TEST
		else
			BUILD_RPMS=$BUILD_RPMS_TEST
		fi
	fi
fi

echo BUILD_RPMS: $BUILD_RPMS

ERROR=0

# Hier gehts los:

if [ -z "$1" ]; then
	# without parameter, we are in source directory
	SPECFILE=$(ls *.spec | awk '{print $1}')
	PACKAGE=$(sed -n -e 's/^Name:\W*//pi' <$SPECFILE)
	VERSION=$(sed -n -e 's/^Version:\W*//pi' <$SPECFILE)
elif [ -r "$1" ]; then
	# parameter is src.rpm file
        PACKAGE=`rpm -qp --qf "%{NAME}" $1`
        VERSION=`rpm -qp --qf "%{VERSION}" $1`
fi

echo $PACKAGE $VERSION $BUILD_DIST $BUILD_MODE

if [ -z "$PACKAGE" ]; then
    echo "nothing to build found!"
    exit 1
fi



rm -rf ${BUILDTEMP}/src/${PACKAGE}
mkdir -p ${BUILDTEMP}/src/${PACKAGE}


if [ "$1" ]; then
	EXPECTED_DEST_RPM_FILENAME=${DEST_DIR}/${BUILD_DIST}/`echo $1 | sed 's/\.src\./.*./'`
	if [ -f $EXPECTED_DEST_RPM_FILENAME ]; then
		echo
		echo "Build of \"${PACKAGE}\" unnecessary. Current RPM already at " $EXPECTED_DEST_RPM_FILENAME
		echo
		exit 1
	else
		cp -a "$1" ${BUILDTEMP}/src/${PACKAGE}
		cd ${BUILDTEMP}/src/${PACKAGE}; rpm2cpio $1 | cpio -i
	fi
else
	cp -a $SPECFILE ${BUILDTEMP}/src/${PACKAGE}
	
	# uses Subversion (svn) to generate more information for the RPM
	if [ "$BUILD_USES_SVN" = "yes" ]; then
		echo "receiving subversion information ..."
		# set RELEASE number (last modification in this directory)
		# requires subversion (but connection to server isn't neccessary)
		SVN_BUILD_RELEASE=$(unset LANG; svn info . | sed -n 's/^Last Changed Rev: \([0-9]*\)/\1/p')
		# in case of local modification a "m" is added to the release number
		SVN_BUILD_MODIFIED=$(svn status -v | grep "^ *M" | wc -l)
		# checks if a file is commited, 
		# but update on its directory is not performed
		SVN_LAST_MODIFICATION=$(svn status -v | sed -n 's/^ *[0-9]* *\([0-9]*\) .*/\1/p' | sort -n | tail -1)
		if [ $SVN_BUILD_MODIFIED -gt 0 ]; then  
			SVN_BUILD_RELEASE=${SVN_BUILD_RELEASE}m
		elif [ $SVN_LAST_MODIFICATION -ne $SVN_BUILD_RELEASE ]; then
			SVN_BUILD_RELEASE=${SVN_BUILD_RELEASE}p
		else # check if RPM has been build already
			EXPECTED_DEST_RPM_FILENAME=${DEST_DIR}/${BUILD_DIST}/${PACKAGE}-${VERSION}-${SVN_BUILD_RELEASE}.*.rpm
			if [ -f $EXPECTED_DEST_RPM_FILENAME ]; then
				echo
				echo "Build of \"${PACKAGE}\" unnecessary. Current RPM already at " $EXPECTED_DEST_RPM_FILENAME
				echo
				exit 1
			fi			
		fi
		if [ "$SVN_BUILD_RELEASE" ]; then
			sed "s/^Release:.*/Release: $SVN_BUILD_RELEASE/ig" $SPECFILE > ${BUILDTEMP}/src/${PACKAGE}/$SPECFILE
		fi
		echo "SVN_BUILD_RELEASE: $SVN_BUILD_RELEASE"
		
		# changelog header
		DATE=`unset LANG; date +"%a %b %d %Y"`
		echo -e "* $DATE -\n" > Changes
		echo "- Subversion status:" >> Changes
		# additional infos for the Changes file (see if everything is checked in)
		svn status -v | grep -v '^? ' >> Changes
		echo >> Changes
		echo "- ChangeLog:" >> Changes
		# requires connection to the subversion server
		svn log -v . > ChangeLog.new && mv ChangeLog.new ChangeLog
		cat ChangeLog >> Changes
		
		# if there is no changelog section in the spec file,
		# also append the Changes there
		if ! grep -q -i "%changelog" $SPECFILE ; then
			# paste "Changes" into SPEC file. 
			echo "%changelog" >> ${BUILDTEMP}/src/${PACKAGE}/$SPECFILE
			# without trailing *, except for the initial
			echo -n "*" >> ${BUILDTEMP}/src/${PACKAGE}/$SPECFILE
			cat Changes | sed 's/\(^[\*]\+\)//g' >> ${BUILDTEMP}/src/${PACKAGE}/$SPECFILE
		fi
	fi
	# just make sure, that the Changes file exist
	touch Changes
	
	# if "SOURCES" file exist, create tar file on the fly
	if [ -r "SOURCES" ]; then
		# create a tar file from the sources
		tar cvjf ${BUILDTEMP}/src/${PACKAGE}/$PACKAGE-$VERSION.tar.bz2 --files-from SOURCES --exclude .svn --exclude "*~"
	
	elif [ -r "SPECSOURCES" ]; then
		# SPECSOURCES has to list the files 
		# that are referenced in the .spec file
		# these files are copied to the build environment
		cp -av `cat SPECSOURCES` ${BUILDTEMP}/src/${PACKAGE}
	
	else 
		# legacy method: 
		# call the existing Makefile 
		# to generate a tar file	
		test -e Makefile && make
		TARFILE=$(ls  *.tar.[gb]z*| awk '{print $1}')
		cp -av $TARFILE ${BUILDTEMP}/src/${PACKAGE}
	fi
fi

# create file for buildvars (see $BUILDSCRIPT)
cat > $BUILDVARS <<EOF
# Variables for build-Prozess (export is needed in buildscript)
BUILD_ROOT=$BUILD_ROOT
BUILD_DIST=$BUILD_DIST
BUILD_RPMS=${DEST_DIR}/${BUILD_DIST}:$BUILD_RPMS

# Variables for execute the build-proggy
BUILD_MODE="$BUILD_MODE"
BUILD_PLACE=${BUILDTEMP}/src/${PACKAGE}
EXTRA_RPMS="$EXTRA_RPMS"
EOF

# starting Suse build (as root)
sudo $BUILDSCRIPT $BUILDVARS
#rm -f $BUILDVARS



mkdir -p ${DEST_DIR}/src
# delete all old versions of this package
for i in ${DEST_DIR}/src/${PACKAGE}-*.src.rpm; do
	if [ -f $i -a ${PACKAGE} = "`rpm -qp --qf "%{NAME}\n" $i`" ]; then
		rm $i
	fi
done
#rm -f ${DEST_DIR}/src/${PACKAGE}-*.src.rpm

if !(cp -a ${BUILD_ROOT}/usr/src/packages/SRPMS/${PACKAGE}-${VERSION}-*.src.rpm ${DEST_DIR}/src 2> /dev/null); then 
	ERROR=4
fi

mkdir -p ${DEST_DIR}/${BUILD_DIST}
# delete all old versions of this package
for i in ${DEST_DIR}/${BUILD_DIST}/${PACKAGE}-*.rpm; do
	if [ -f $i -a ${PACKAGE} = "`rpm -qp --qf "%{NAME}\n" $i`" ]; then
		rm $i
	fi
done
#rm -f ${DEST_DIR}/${BUILD_DIST}/${PACKAGE}-*.rpm

if !(cp -a ${BUILD_ROOT}/usr/src/packages/RPMS/*/${PACKAGE}*${VERSION}-*.*.rpm ${DEST_DIR}/${BUILD_DIST} ); then 
	ERROR=8
fi

echo
if [ $ERROR -ne 0 ]; then

	echo "ERROR while building packet $PACKAGE";
else
	echo "Build of \"${PACKAGE}\" completed!"
	echo "Packages can be found in ${DEST_DIR}"
fi
echo

exit $ERROR
